<?php

namespace App\Http\Controllers;

use App\Models\Cart;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    private function getCartItems()
    {
        if (Auth::check()) {
            // Authenticated user - get from database
            return Cart::where('user_id', Auth::id())
                ->with(['product', 'size'])
                ->get()
                ->map(function($item) {
                    return [
                        'id' => $item->id,
                        'product_id' => $item->product_id,
                        'size_id' => $item->size_id,
                        'product' => $item->product,
                        'size' => $item->size,
                        'quantity' => $item->quantity,
                        'custom_measurements' => $item->custom_measurements,
                    ];
                });
        } else {
            // Guest user - get from session
            $sessionCart = session('cart', []);
            $items = [];
            
            foreach ($sessionCart as $key => $cartItem) {
                if (is_array($cartItem)) {
                    $product = Product::find($cartItem['product_id']);
                    $size = isset($cartItem['size_id']) ? \App\Models\Size::find($cartItem['size_id']) : null;
                    if ($product) {
                        $items[] = [
                            'id' => 'session_' . $key,
                            'product_id' => $cartItem['product_id'],
                            'size_id' => $cartItem['size_id'] ?? null,
                            'product' => $product,
                            'size' => $size,
                            'quantity' => $cartItem['quantity'],
                            'custom_measurements' => $cartItem['custom_measurements'] ?? null,
                        ];
                    }
                } else {
                    // Backward compatibility with old session format
                    $product = Product::find($key);
                    if ($product) {
                        $items[] = [
                            'id' => 'session_' . $key,
                            'product_id' => $key,
                            'size_id' => null,
                            'product' => $product,
                            'size' => null,
                            'quantity' => $cartItem,
                        ];
                    }
                }
            }
            
            return collect($items);
        }
    }

    public function index()
    {
        // Prevent admins from accessing cart
        if (Auth::check() && Auth::user()->isAdmin()) {
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط. لا يمكنك إضافة منتجات للسلة.' 
                    : 'Admin account is for store management only. You cannot add items to cart.');
        }

        $cartItems = $this->getCartItems();
        
        $total = $cartItems->sum(function($item) {
            $itemPrice = $item['product']->price;
            
            // إذا كان التصنيف 8 واختار العميل تفصيل، أضف 5 ريال
            if ($item['product']->category_id == 8 && isset($item['custom_measurements']) && $item['custom_measurements']) {
                $itemPrice += 5;
            }
            
            return $item['quantity'] * $itemPrice;
        });
        
        return view('cart.index', compact('cartItems', 'total'));
    }

    public function store(Request $request)
    {
        // Prevent admins from adding items to cart
        if (Auth::check() && Auth::user()->isAdmin()) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => app()->getLocale() === 'ar' 
                        ? 'حساب المدير مخصص لإدارة المتجر فقط.' 
                        : 'Admin account is for store management only.',
                ], 403);
            }
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط. لا يمكنك إضافة منتجات للسلة.' 
                    : 'Admin account is for store management only. You cannot add items to cart.');
        }

        try {
            $request->validate([
                'product_id' => 'required|exists:products,id',
                'size_id' => 'nullable|exists:sizes,id',
                'quantity' => 'required|integer|min:1',
                'is_custom_order' => 'nullable|boolean',
                'custom_measurements' => 'nullable|string',
                'cart_item_id' => 'nullable|string',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage(),
                    'errors' => $e->errors(),
                ], 422);
            }
            throw $e;
        }

        try {
            $product = Product::with('sizes')->findOrFail($request->product_id);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'المنتج غير موجود',
                ], 404);
            }
            throw $e;
        }
        
        // معالجة بيانات القياسات المخصصة (إن وجدت)
        $customMeasurementsArray = null;
        $isCustomOrder = filter_var($request->input('is_custom_order'), FILTER_VALIDATE_BOOLEAN);
        if ($isCustomOrder && $request->filled('custom_measurements')) {
            $decoded = json_decode($request->input('custom_measurements'), true);
            if (is_array($decoded)) {
                $customMeasurementsArray = $decoded;
            }
        }

        // التحقق من القياس إذا كان المنتج يحتوي على قياسات (وليس طلب تفصيل)
        $availableStock = $product->stock;
        if ($product->sizes->count() > 0 && !$isCustomOrder) {
            // إذا كان المنتج له قياسات وليس طلب تفصيل، يجب اختيار قياس
            if (!$request->size_id) {
                if ($request->ajax() || $request->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'يرجى اختيار قياس للمنتج',
                    ], 400);
                }
                return back()->with('error', 'يرجى اختيار قياس للمنتج');
            }
            
            $selectedSize = $product->sizes->where('id', $request->size_id)->first();
            if (!$selectedSize) {
                if ($request->ajax() || $request->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'القياس المحدد غير متوفر لهذا المنتج',
                    ], 400);
                }
                return back()->with('error', 'القياس المحدد غير متوفر لهذا المنتج');
            }
            
            $availableStock = $selectedSize->pivot->stock ?? 0;
        } elseif ($isCustomOrder) {
            // إذا كان طلب تفصيل، نستخدم المخزون الأساسي للمنتج
            $availableStock = $product->stock;
        }
        
        // التحقق من المخزون المتاح
        if ($availableStock < $request->quantity) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.out_of_stock'),
                ], 400);
            }
            return back()->with('error', __('messages.out_of_stock'));
        }

        $cartItemId = $request->input('cart_item_id');

        if (Auth::check()) {
            // Authenticated user - save to database
            if ($cartItemId) {
                // تعديل عنصر موجود من السلة
                $cartItem = Cart::where('user_id', Auth::id())->find($cartItemId);
                if ($cartItem) {
                    $cartItem->update([
                        'product_id' => $request->product_id,
                        'size_id' => $request->size_id,
                        'quantity' => $request->quantity,
                        'custom_measurements' => $customMeasurementsArray,
                    ]);
                } else {
                    // إذا لم يتم العثور عليه نرجع للسلوك الافتراضي (إضافة / تحديث)
                    Cart::updateOrCreate(
                        [
                            'user_id' => Auth::id(),
                            'product_id' => $request->product_id,
                            'size_id' => $request->size_id,
                        ],
                        [
                            'quantity' => $request->quantity,
                            'custom_measurements' => $customMeasurementsArray,
                        ]
                    );
                }
            } else {
                // سلوك الإضافة العادي
                Cart::updateOrCreate(
                    [
                        'user_id' => Auth::id(),
                        'product_id' => $request->product_id,
                        'size_id' => $request->size_id,
                    ],
                    [
                        'quantity' => $request->quantity,
                        'custom_measurements' => $customMeasurementsArray,
                    ]
                );
            }
        } else {
            // Guest user - save to session
            $cart = session('cart', []);
            
            // إنشاء مفتاح فريد: product_id + size_id (أو 'custom' إذا كان تفصيل) + hash من القياسات المخصصة
            if ($isCustomOrder && $customMeasurementsArray) {
                $cmHash = md5(json_encode($customMeasurementsArray));
                $newKey = $request->product_id . '_custom_' . $cmHash;
            } else {
                $newKey = $request->product_id . '_' . ($request->size_id ?? 'no_size');
            }
            
            // إذا كنا نعدل عنصر موجود من السلة
            if ($cartItemId && str_starts_with($cartItemId, 'session_')) {
                $oldKey = str_replace('session_', '', $cartItemId);

                // إذا تغيّر المفتاح (مثلاً تغيّر القياس أو القياسات المخصصة)، نحذف القديم
                if ($oldKey !== $newKey && isset($cart[$oldKey])) {
                    unset($cart[$oldKey]);
                }

                $cart[$newKey] = [
                    'product_id' => $request->product_id,
                    'size_id' => $request->size_id,
                    'quantity' => $request->quantity,
                    'custom_measurements' => $customMeasurementsArray,
                ];
            } else {
                // سلوك الإضافة العادي (زيادة الكمية)
                $existingQuantity = isset($cart[$newKey]['quantity']) ? (int) $cart[$newKey]['quantity'] : 0;
                $cart[$newKey] = [
                    'product_id' => $request->product_id,
                    'size_id' => $request->size_id,
                    'quantity' => $existingQuantity + $request->quantity,
                    'custom_measurements' => $customMeasurementsArray,
                ];
            }
            session(['cart' => $cart]);
        }

        // حساب عدد العناصر في السلة بعد الإضافة
        if (Auth::check()) {
            $cartCount = Cart::where('user_id', Auth::id())->sum('quantity');
        } else {
            $sessionCart = session('cart', []);
            if (empty($sessionCart)) {
                $cartCount = 0;
            } else {
                $first = reset($sessionCart);
                if (is_array($first)) {
                    // الشكل الجديد: كل عنصر مصفوفة تحتوي على الكمية
                    $cartCount = collect($sessionCart)->sum(function ($item) {
                        return $item['quantity'] ?? 0;
                    });
                } else {
                    // الشكل القديم: product_id => quantity
                    $cartCount = array_sum($sessionCart);
                }
            }
        }

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success'    => true,
                'message'    => __('messages.add_to_cart'),
                'cart_count' => $cartCount,
            ]);
        }

        return back()->with('success', __('messages.add_to_cart'));
    }

    public function update(Request $request, $id)
    {
        // Prevent admins from updating cart
        if (Auth::check() && Auth::user()->isAdmin()) {
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط.' 
                    : 'Admin account is for store management only.');
        }

        $request->validate([
            'quantity' => 'required|integer|min:1',
        ]);

        if (Auth::check()) {
            // Authenticated user
            $cartItem = Cart::where('user_id', Auth::id())->findOrFail($id);
            
            if ($cartItem->product->stock < $request->quantity) {
                return back()->with('error', __('messages.out_of_stock'));
            }

            $cartItem->update(['quantity' => $request->quantity]);
        } else {
            // Guest user - update session
            if (str_starts_with($id, 'session_')) {
                $productId = str_replace('session_', '', $id);
                $product = Product::findOrFail($productId);
                
                if ($product->stock < $request->quantity) {
                    return back()->with('error', __('messages.out_of_stock'));
                }
                
                $cart = session('cart', []);
                $cart[$productId] = $request->quantity;
                session(['cart' => $cart]);
            }
        }

        return back()->with('success', __('messages.update_cart'));
    }

    public function destroy($id)
    {
        // Prevent admins from removing items from cart
        if (Auth::check() && Auth::user()->isAdmin()) {
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط.' 
                    : 'Admin account is for store management only.');
        }

        if (Auth::check()) {
            // Authenticated user
            $cartItem = Cart::where('user_id', Auth::id())->findOrFail($id);
            $cartItem->delete();
        } else {
            // Guest user - remove from session
            if (str_starts_with($id, 'session_')) {
                $productId = str_replace('session_', '', $id);
                $cart = session('cart', []);
                unset($cart[$productId]);
                session(['cart' => $cart]);
            }
        }

        return back()->with('success', __('messages.remove_from_cart'));
    }
}
