<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Cart;
use App\Mail\WelcomeEmail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rules;

class RegisteredUserController extends Controller
{
    public function create(Request $request)
    {
        $redirect = $request->get('redirect', route('home'));
        return view('auth.register', compact('redirect'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'points' => 0,
        ]);

        Auth::login($user);

        // Merge session cart with database cart after registration
        $sessionCart = session('cart', []);
        if (!empty($sessionCart)) {
            foreach ($sessionCart as $key => $cartItem) {
                if (is_array($cartItem)) {
                    // الشكل الجديد: مصفوفة تحتوي على product_id, size_id, quantity, custom_measurements
                    $productId = $cartItem['product_id'] ?? null;
                    $sizeId = $cartItem['size_id'] ?? null;
                    $quantity = $cartItem['quantity'] ?? 1;
                    $customMeasurements = $cartItem['custom_measurements'] ?? null;
                    
                    if ($productId) {
                        Cart::updateOrCreate(
                            [
                                'user_id' => Auth::id(),
                                'product_id' => $productId,
                                'size_id' => $sizeId,
                            ],
                            [
                                'quantity' => $quantity,
                                'custom_measurements' => $customMeasurements,
                            ]
                        );
                    }
                } else {
                    // الشكل القديم: product_id => quantity (للتوافق مع السلة القديمة)
                    $productId = $key;
                    $quantity = $cartItem;
                    
                    if (is_numeric($productId)) {
                        Cart::updateOrCreate(
                            [
                                'user_id' => Auth::id(),
                                'product_id' => $productId,
                                'size_id' => null,
                            ],
                            [
                                'quantity' => $quantity,
                            ]
                        );
                    }
                }
            }
            session()->forget('cart');
        }

        // Send welcome email
        try {
            Mail::to($user->email)->send(new WelcomeEmail($user));
        } catch (\Exception $e) {
            // Log error but don't fail registration
            \Log::error('Failed to send welcome email: ' . $e->getMessage());
        }

        // Redirect to intended URL or checkout if coming from cart
        $redirect = $request->input('redirect');
        
        // إذا كان الـ redirect يشير إلى صفحة متابعة الطلب، أو إذا كان المستخدم قادم من السلة
        if ($redirect) {
            return redirect($redirect);
        }
        
        // التحقق من الـ intended URL (إذا كان المستخدم حاول الوصول لصفحة متابعة الطلب)
        $intended = session()->pull('url.intended');
        if ($intended && str_contains($intended, 'orders.create')) {
            return redirect(route('orders.create'));
        }
        
        // إذا كان هناك منتجات في السلة، نوجهه لمتابعة الطلب
        $cartCount = Cart::where('user_id', Auth::id())->sum('quantity');
        if ($cartCount > 0) {
            return redirect(route('orders.create'));
        }

        return redirect(route('home'));
    }
}
