<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Cart;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class AuthenticatedSessionController extends Controller
{
    public function create(Request $request)
    {
        $redirect = $request->get('redirect', route('home'));
        return view('auth.login', compact('redirect'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (! Auth::attempt($request->only('email', 'password'), $request->boolean('remember'))) {
            throw ValidationException::withMessages([
                'email' => __('auth.failed'),
            ]);
        }

        $request->session()->regenerate();

        // Merge session cart with database cart after login
        $sessionCart = session('cart', []);
        if (!empty($sessionCart)) {
            foreach ($sessionCart as $key => $cartItem) {
                if (is_array($cartItem)) {
                    // الشكل الجديد: مصفوفة تحتوي على product_id, size_id, quantity, custom_measurements
                    $productId = $cartItem['product_id'] ?? null;
                    $sizeId = $cartItem['size_id'] ?? null;
                    $quantity = $cartItem['quantity'] ?? 1;
                    $customMeasurements = $cartItem['custom_measurements'] ?? null;
                    
                    if ($productId) {
                        Cart::updateOrCreate(
                            [
                                'user_id' => Auth::id(),
                                'product_id' => $productId,
                                'size_id' => $sizeId,
                            ],
                            [
                                'quantity' => $quantity,
                                'custom_measurements' => $customMeasurements,
                            ]
                        );
                    }
                } else {
                    // الشكل القديم: product_id => quantity (للتوافق مع السلة القديمة)
                    $productId = $key;
                    $quantity = $cartItem;
                    
                    if (is_numeric($productId)) {
                        Cart::updateOrCreate(
                            [
                                'user_id' => Auth::id(),
                                'product_id' => $productId,
                                'size_id' => null,
                            ],
                            [
                                'quantity' => $quantity,
                            ]
                        );
                    }
                }
            }
            session()->forget('cart');
        }

        // Redirect to intended URL or checkout if coming from cart
        $redirect = $request->input('redirect');
        
        // إذا كان الـ redirect يشير إلى صفحة متابعة الطلب
        if ($redirect) {
            return redirect($redirect);
        }
        
        // التحقق من الـ intended URL (إذا كان المستخدم حاول الوصول لصفحة متابعة الطلب)
        $intended = session()->pull('url.intended');
        if ($intended && str_contains($intended, 'orders.create')) {
            return redirect(route('orders.create'));
        }
        
        // إذا كان هناك منتجات في السلة، نوجهه لمتابعة الطلب
        $cartCount = Cart::where('user_id', Auth::id())->sum('quantity');
        if ($cartCount > 0) {
            return redirect(route('orders.create'));
        }

        return redirect()->intended(route('home'));
    }

    public function destroy(Request $request)
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }
}
